#!/bin/sh

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/syno/sbin:/usr/syno/bin

IPTABLES="iptables"
MODULES_BASE="/lib/modules"
RULES_IPTABLES="/etc/firewall_rules.dump"
RULES_6_IPTABLES="/etc/firewall_6_rules.dump"
KERNEL_VERSION=`uname -r`
PROC_IPTABLES_NAMES="/proc/net/ip_tables_names"
PROC_6_IPTABLES_NAMES="/proc/net/ip6_tables_names"
V4ONLY=`get_key_value /etc.defaults/synoinfo.conf ipv4only`
WIRELESS_INFO="/tmp/wireless.info"
WIRELESS_AP="/usr/syno/etc/wireless_ap.conf"
KERNEL_FW_STATUS_FILE="/proc/sys/kernel/syno_firewall_status"
SZD_FW_SECURITY_ROOT="/etc/fw_security"
FW_SECURITY_SCRIPT="$SZD_FW_SECURITY_ROOT/sysconf/iptables_security.sh"
FW="firewall"
NAT="nat"
NATTEST="nattest"
PORTFW="portfw"
HAS_PORTFW_RULE=0

# For guest wifi
RC_NETWORK=/etc/rc.network

case "$KERNEL_VERSION" in
"2.4.22-uc0")
	KERNEL_MODULES_CORE="ip_tables.o iptable_filter.o ipt_LOG.o"
	KERNEL_MODULES_COMMON="ip_conntrack.o ipt_state.o ipt_multiport.o iptable_nat.o ipt_REDIRECT.o ipt_limit.o"
	IPV6_MODULES=""
	;;
"2.6.15")
	KERNEL_MODULES_CORE="ip_tables.ko iptable_filter.ko ip_conntrack.ko ip_nat.ko"
	KERNEL_MODULES_COMMON="ipt_multiport.ko ipt_state.ko"
	KERNEL_MODULES_NAT="iptable_nat.ko ipt_REDIRECT.ko"
	IPV6_MODULES=""
	;;
"2.6.24")
	KERNEL_MODULES_CORE="x_tables.ko ip_tables.ko iptable_filter.ko nf_conntrack.ko nf_conntrack_ipv4.ko ipt_LOG.ko"
	KERNEL_MODULES_COMMON="xt_multiport.ko xt_tcpudp.ko xt_state.ko xt_limit.ko"
	KERNEL_MODULES_NAT="nf_nat.ko iptable_nat.ko ipt_REDIRECT.ko"
	IPV6_MODULES="x_tables.ko nf_conntrack.ko ip6_tables.ko ip6table_filter.ko nf_conntrack_ipv6.ko ip6t_LOG.ko"
	;;
"2.6.3"[2-6]*)
	KERNEL_MODULES_CORE="x_tables.ko ip_tables.ko iptable_filter.ko nf_conntrack.ko nf_defrag_ipv4.ko nf_conntrack_ipv4.ko ipt_LOG.ko"
	KERNEL_MODULES_COMMON="xt_multiport.ko xt_tcpudp.ko xt_state.ko xt_limit.ko"
	KERNEL_MODULES_NAT="nf_nat.ko iptable_nat.ko ipt_REDIRECT.ko ipt_MASQUERADE.ko"
	IPV6_MODULES="x_tables.ko nf_conntrack.ko ip6_tables.ko ip6table_filter.ko nf_conntrack_ipv6.ko ip6t_LOG.ko"
	;;
"2.6.3"[7-9]*|"3."*)
	KERNEL_MODULES_CORE="x_tables.ko ip_tables.ko iptable_filter.ko nf_conntrack.ko nf_defrag_ipv4.ko nf_conntrack_ipv4.ko ipt_LOG.ko"
	KERNEL_MODULES_COMMON="xt_multiport.ko xt_tcpudp.ko xt_state.ko xt_limit.ko"
	KERNEL_MODULES_NAT="nf_nat.ko iptable_nat.ko ipt_REDIRECT.ko ipt_MASQUERADE.ko"
	IPV6_MODULES="x_tables.ko nf_conntrack.ko ip6_tables.ko ip6table_filter.ko nf_defrag_ipv6.ko nf_conntrack_ipv6.ko ip6t_LOG.ko"
	;;
*)
	echo "******iptables: Kernel version not supported******"
	;;
esac

success() {
	[ -n $1 ] && echo "success" || echo "$1: success"
}

failure() {
	[ -n $1 ] && echo "failed" || echo "$1: failed"
}

reverse_syno() {
	local modules=$1
	local mod
	local ret=""
	
	for mod in $modules; do
	    ret="$mod $ret"
	done
	
	echo $ret
}

dump_portfwd_rules()
{
	local PF_FILTER_DUMP="/etc/portforward/routerpf/filter_rules.dump"
	local RULE_FILE="$1"
	local TMP_IPTABLES_RULE="$1.saved"

	rm ${PF_FILTER_DUMP} &> /dev/null
	synorouterportfwd
	if [ -f "$PF_FILTER_DUMP" ]; then
		echo "*filter" > ${TMP_IPTABLES_RULE}
		if [ -f "$RULE_FILE" ]; then
			 grep -v "COMMIT" "$RULE_FILE" | grep -v "*filter" >> ${TMP_IPTABLES_RULE}
		fi
		cat ${PF_FILTER_DUMP} >> ${TMP_IPTABLES_RULE}
		echo "COMMIT" >> ${TMP_IPTABLES_RULE}
		mv ${TMP_IPTABLES_RULE} ${RULE_FILE}
		rm ${PF_FILTER_DUMP}
		HAS_PORTFW_RULE=1
		return 0
	fi
	return 1
}

#ipv6 ip6tables
ipv6_start() {
	local RULES_FILE=$1
	local services=$2
	local service=''

	# platform dont support ipv6
	if [ -z "${IPV6_MODULES}" ]; then
		return
	fi

	for service in $services; do
		/usr/syno/bin/iptablestool --insmod $service ${IPV6_MODULES} ${KERNEL_MODULES_COMMON}
	done

	# if ipv6 iptable rules not exist, we skip
	if [ -e ${RULES_FILE} ]; then
		/sbin/ip6tables-restore < ${RULES_FILE}
	else
		/sbin/ip6tables -F
	fi
}

#ipv4 iptables, ipv4 modules must been loaded, because of following ipv6 firewall may use those modules
ipv4_start() {
	local RULES_FILE=$1
	local services=$2
	local service=''

	for service in $services; do
		/usr/syno/bin/iptablestool --insmod $service ${KERNEL_MODULES_CORE} ${KERNEL_MODULES_COMMON}
	done

	if [ -e ${RULES_FILE} ]; then
		/sbin/iptables-restore < ${RULES_FILE}
	else
		/sbin/iptables -F
	fi
}

dump_rules_clear() {
	if [ -e $1 ]; then
		rm -f $1
	fi
	if [ -e $2 ]; then
		rm -f $2
	fi
}

load_nat_module()
{
	local port_forward="$1"

	/usr/syno/bin/iptablestool --checkap
	local ret_checkap=$?

	if [ "${port_forward}" = "forwarding_test" ]; then
		/usr/syno/bin/iptablestool --insmod $NATTEST ${KERNEL_MODULES_CORE} ${KERNEL_MODULES_NAT} ${KERNEL_MODULES_COMMON}
	elif [ 1 -eq $ret_checkap ]; then
		/usr/syno/bin/iptablestool --insmod $NAT ${KERNEL_MODULES_CORE} ${KERNEL_MODULES_NAT}
	fi

	return 0
}

unload_nat_module()
{
	local port_forward="$1"
	local modules_reverse=`reverse_syno "${KERNEL_MODULES_NAT}"`
	local fw_has_rules=0
	local portfw_has_rules=0

	dump_fw_rules "/etc/firewall" "0"
	fw_has_rules=$?
	dump_portfwd_rules ${RULES_IPTABLES}
	portfw_has_rules=$?

	if [ "${port_forward}" = "forwarding_test" ]; then
		if [ 0 -eq $fw_has_rules -o 0 -eq $portfw_has_rules ]; then
			stop_kernel $NATTEST
		else
			stop $NATTEST 1
		fi
	else
		#Run AP mode
		if [ 0 -eq $fw_has_rules -o 0 -eq $portfw_has_rules ]; then
			stop_kernel $NAT
		else
			stop $NAT 1
		fi
	fi
}

dump_fw_rules()
{
	local v6rules="${RULES_6_IPTABLES}"
	local ret=
	[ "yes" = "${V4ONLY}" ] && v6rules="none"

	dump_rules_clear ${RULES_IPTABLES} ${RULES_6_IPTABLES}
	firewalltool -dump_rules $1 ${RULES_IPTABLES} ${v6rules} $2
	ret=$?

	[ -f "${RULES_IPTABLES}" -o -f "${RULES_6_IPTABLES}" ] && return 0
	return 1
}

start() {
	local ret=0
	local is_deny_stateful_conn=0
	local service=''
	local is_no_rec=0

	if [ $1 -eq 1 ];then
		is_deny_stateful_conn=1
	else
		is_deny_stateful_conn=0
	fi

	if [ -z $2 ];then
		RULES_DIR="/etc/firewall"
	else
		RULES_DIR=$2
	fi

	dump_fw_rules ${RULES_DIR} ${is_deny_stateful_conn}
	if [ 0 -ne $? ]; then
		stop $FW 1
		is_no_rec=1
	else
		service="$service $FW"
	fi

	dump_portfwd_rules ${RULES_IPTABLES}
	if [ 0 -ne $? ]; then
		if [ 1 -eq $is_no_rec ]; then
			stop $PORTFW 1
			return 0
		else
			stop $PORTFW 0
		fi
	else
		service="$service $PORTFW"
	fi

	[ -f ${RULES_IPTABLES} ] && ipv4_start ${RULES_IPTABLES} ${service}
	[ -f ${RULES_6_IPTABLES} ] && ipv6_start ${RULES_6_IPTABLES} ${service}

	return 0
}

ipv6_stop_kernel() {
	local modules_reverse=""

	# ipv6 Firewall modules not exists
	if [ ! -e "$PROC_6_IPTABLES_NAMES" ];then
		return 1;
	fi

	echo ""
	echo "Unloading kernel ipv6 netfilter modules... "
	modules_reverse=`reverse_syno "$IPV6_MODULES $KERNEL_MODULES_COMMON"`
	/usr/syno/bin/iptablestool --rmmod $1 $modules_reverse
}

#ipv6 ip6tables -F
ipv6_stop() {
	local modules_reverse=""
	local rmcfg=$2

	# ipv6 Firewall modules not exists
	if [ ! -e "$PROC_6_IPTABLES_NAMES" ];then
		return 1;
	fi

	echo -n "Clean all ipv6 firewall rules... "
	/sbin/ip6tables -F
	[ $? -eq 0 ] && success || failure

	ipv6_stop_kernel $1

	if [ 1 -eq $rmcfg ]; then
		rm -f $RULES_6_IPTABLES &> /dev/null
	fi
}

ipv4_stop_kernel() {
	local modules_reverse=""

	if [ ! -e "$PROC_IPTABLES_NAMES" ];then
		return 1
	fi

	if [ $NAT == $1 ]; then
		echo ""
		echo "Unloading kernel ipv4 netfilter NAT modules... "
		modules_reverse=`reverse_syno "$KERNEL_MODULES_NAT"`
		/usr/syno/bin/iptablestool --rmmod $1 $modules_reverse
	elif [ $NATTEST == $1 ]; then
		echo ""
		echo "Unloading kernel ipv4 netfilter NAT & Common modules... "
		modules_reverse=`reverse_syno "$KERNEL_MODULES_COMMON $KERNEL_MODULES_NAT"`
		/usr/syno/bin/iptablestool --rmmod $1 $modules_reverse
	else #$FW, $PORTFW
		echo ""
		echo "Unloading kernel ipv4 netfilter common modules... "
		modules_reverse=`reverse_syno "$KERNEL_MODULES_COMMON"`
		/usr/syno/bin/iptablestool --rmmod $1 $modules_reverse
	fi

	echo ""
	echo "Unloading kernel ipv4 netfilter core modules... "
	modules_reverse=`reverse_syno "$KERNEL_MODULES_CORE"`
	/usr/syno/bin/iptablestool --rmmod $1 $modules_reverse
}

#ipv4 iptables -F
ipv4_stop() {
	local modules_reverse=""
	local rmcfg=$2

	if [ ! -e "$PROC_IPTABLES_NAMES" ];then
		return 1
	fi

	echo -n "Clean all ipv4 firewall rules... "
	/sbin/iptables -F
	[ $? -eq 0 ] && success || failure

	ipv4_stop_kernel $1

	if [ 1 -eq $rmcfg ]; then
		rm -f $RULES_IPTABLES &> /dev/null
	fi
}

stop_kernel() {
	# stop ipv6 first then ipv4 ~ because of the modules loaded sequence

	[ "yes" != "${V4ONLY}" ] && ipv6_stop_kernel $1
	ipv4_stop_kernel $1

	return $ret
}

stop() {
	# stop ipv6 first then ipv4 ~ because of the modules loaded sequence
	local services=''
	local serv=''
	local rmcfg=1

	if [ -z $1 ];then
		services="$NAT $NATTEST $FW $PORTFW"
	else
		services=$1
	fi

	if [ -z $2 ];then
		rmcfg=1
	else
		rmcfg=$2
	fi

	for serv in $services; do
		[ "yes" != "${V4ONLY}" ] && ipv6_stop $serv $rmcfg
		ipv4_stop $serv $rmcfg
	done

	return $ret
}

restart() {
	stop
	start 0 $1
}

case "$1" in
	start)
		start 0 $2
		RETVAL=$?
		${RC_NETWORK} start-guest-net-access-rule
		$FW_SECURITY_SCRIPT start $HAS_PORTFW_RULE
		;;
	force-reload)
		start $2 $3
		RETVAL=$?
		${RC_NETWORK} start-guest-net-access-rule
		$FW_SECURITY_SCRIPT start $HAS_PORTFW_RULE
		;;
	stop)
		stop
		RETVAL=$?
		${RC_NETWORK} stop-guest-net-access-rule
		$FW_SECURITY_SCRIPT stop
		;;
	restart)
		restart $2
		RETVAL=$?
		${RC_NETWORK} stop-guest-net-access-rule
		${RC_NETWORK} start-guest-net-access-rule
		$FW_SECURITY_SCRIPT restart $HAS_PORTFW_RULE
		;;
	load_nat_mod)
		load_nat_module $2
		RETVAL=$?
		;;
	unload_nat_mod)
		unload_nat_module $2
		${RC_NETWORK} stop-guest-net-access-rule
		${RC_NETWORK} start-guest-net-access-rule
		$FW_SECURITY_SCRIPT restart $HAS_PORTFW_RULE
		;;
	*)
		echo $"Usage: ${0##*/} {start|stop|restart|force-reload|load_nat_mod|unload_nat_mod}"
		RETVAL=2
		;;
esac

exit $RETVAL
